<?php
/**
 * Stock Controller
 * File ini menangani proses upload dan manajemen data saham
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/session.php';
require_once __DIR__ . '/../model/stock_model.php';
require_once __DIR__ . '/../utils/csv_handler.php';

// Definisikan BASE_URL jika belum didefinisikan
if (!defined('BASE_URL')) {
    define('BASE_URL', '/bri_stock_prediction');
}

/**
 * Menangani proses upload file CSV
 */
function handleUpload() {
    // Cek apakah user sudah login
    if (!isLoggedIn()) {
        header('Location: ' . BASE_URL . '/view/auth/login.php');
        exit;
    }
    
    // Cek apakah form sudah disubmit
    if ($_SERVER['REQUEST_METHOD'] == 'POST') {
        // Cek apakah ada file yang diupload
        if (!isset($_FILES['csv_file']) || $_FILES['csv_file']['error'] != UPLOAD_ERR_OK) {
            setFlashMessage('error', 'Gagal mengupload file');
            return;
        }
        
        $file = $_FILES['csv_file'];
        
        // Validasi tipe file
        $fileType = pathinfo($file['name'], PATHINFO_EXTENSION);
        if ($fileType != 'csv') {
            setFlashMessage('error', 'File harus berformat CSV');
            return;
        }
        
        // Buat direktori upload jika belum ada
        $uploadDir = __DIR__ . '/../uploads/';
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        // Tentukan nama file
        $filename = 'stock_data_' . date('YmdHis') . '.csv';
        $targetFile = $uploadDir . $filename;
        
        // Upload file
        if (move_uploaded_file($file['tmp_name'], $targetFile)) {
            // Baca dan validasi data CSV
            $data = readCSV($targetFile);
            $validatedData = validateStockData($data);
            
            if ($validatedData) {
                // Simpan data ke database
                if (saveStockData($validatedData)) {
                    setFlashMessage('success', 'File berhasil diupload dan data berhasil disimpan');
                    
                    // Redirect ke halaman view dengan BASE_URL
                    header('Location: ' . BASE_URL . '/view/stock/view.php');
                    exit;
                } else {
                    setFlashMessage('error', 'Gagal menyimpan data ke database');
                }
            } else {
                setFlashMessage('error', 'Format data CSV tidak valid');
                
                // Hapus file jika data tidak valid
                unlink($targetFile);
            }
        } else {
            setFlashMessage('error', 'Gagal memindahkan file yang diupload');
        }
    }
}

/**
 * Mendapatkan data saham dengan pagination
 * 
 * @param int $page Halaman saat ini
 * @param int $limit Batas data per halaman
 * @return array Data saham dan informasi pagination
 */
function getStockDataPaginated($page = 1, $limit = 10) {
    $page = max(1, $page);
    $offset = ($page - 1) * $limit;
    
    $data = getAllStockData($limit, $offset);
    $totalData = getStockDataCount();
    $totalPages = ceil($totalData / $limit);
    
    return [
        'data' => $data,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total_data' => $totalData,
            'total_pages' => $totalPages
        ]
    ];
}

/**
 * Menangani proses reset data
 */
function handleReset() {
    // Cek apakah user sudah login
    if (!isLoggedIn()) {
        header('Location: ' . BASE_URL . '/view/auth/login.php');
        exit;
    }
    
    // Cek apakah form sudah disubmit
    if ($_SERVER['REQUEST_METHOD'] == 'POST') {
        if (isset($_POST['confirm_reset']) && $_POST['confirm_reset'] == 'yes') {
            // Hapus semua data saham
            if (deleteAllStockData()) {
                setFlashMessage('success', 'Semua data saham berhasil dihapus');
                
                // Redirect ke halaman view dengan BASE_URL
                header('Location: ' . BASE_URL . '/view/stock/view.php');
                exit;
            } else {
                setFlashMessage('error', 'Gagal menghapus data saham');
            }
        } else {
            setFlashMessage('error', 'Konfirmasi penghapusan diperlukan');
        }
    }
}

/**
 * Mendapatkan data untuk visualisasi
 * 
 * @param int $limit Batas jumlah data
 * @return array Data untuk visualisasi
 */
function getVisualizationData($limit = 100) {
    return getChartData($limit);
}